//----------------------------------------------------------------------------//
//- CaoCheck.pas -------------------------------------------------------------//
//----------------------------------------------------------------------------//

//- Beschreibung -------------------------------------------------------------//
// Stellt eine Klasse zur Verfgung, die Eingaben bei Dialogen und Frames     //
// berprft                                                                  //
//----------------------------------------------------------------------------//

unit CaoCheck;

//------------------------------------------------------------------------------

interface

//------------------------------------------------------------------------------

uses
  Forms, Controls, Dialogs, StdCtrls, SysUtils, ComCtrls, CaoRegistry;

//------------------------------------------------------------------------------

Type
  TCaoCheck = class(TObject)

  public
    constructor Create;  reintroduce;
    destructor  Destroy; reintroduce;
    function    CheckFormData           (obForm     : TForm)    : Boolean;
    function    ListViewItemIsSelected  (obListView : TListView): Boolean;
    function    CheckPassword           (sPassword  : string)   : Boolean;

    function  DeleteLettersInString   (var sString: string; fDotAllowed: Boolean): Boolean;
    procedure EditWithNumbers         (var obEdit: TEdit; fDotAllowed: Boolean);
    procedure InitEditsInForms        (var obForm: TForm);

    //--------------------------------------------------------------------------

  private
    m_obRegistry: TCaoRegistry;

    //--------------------------------------------------------------------------

    function PrvCheckLabel(obLabel: TLabel): Boolean;

  //----------------------------------------------------------------------------

  end; // TCaoCheck

//------------------------------------------------------------------------------

implementation

//------------------------------------------------------------------------------

uses
  CamConst;

//------------------------------------------------------------------------------

function TCaoCheck.CheckFormData(obForm: TForm): Boolean;
(*
  CheckFormData prft alle Edit-Felder eines Formulars, ob alle mit "*"
  gekennzeichneten Muss-Eingaben gemacht wurden
*)
var
  iFormIndex,
  iFrameIndex : Integer;
  obTempFrame : TFrame;
begin
  // verfgt das Formular ber keine Edit-Felder wird True zurckgegeben
  Result := True;

  // Alle Komponenten des Formulars durchlaufen
  for iFormIndex := 0 to obForm.ComponentCount - 1 do begin
    // Handelt es sich bei der aktuellen Komponente um ein Label -> weitere Prfung
    if (obForm.Components[iFormIndex] is TLabel) then begin
      Result := PrvCheckLabel(obForm.Components[iFormIndex] as TLabel);
      if not Result then begin
        Exit;
      end; // if
    end // if

    // Handelt es sich bei der aktuellen Komponente um ein Frame, wird das Frame abgesucht
    else if (obForm.Components[iFormIndex] is TFrame) then begin
      // Temporre Framevariable initialisieren
      obTempFrame := (obForm.Components[iFormIndex] as TFrame);
      // Alle Komponenten des Frames durchlaufen
      for iFrameIndex := 0 to obTempFrame.ComponentCount - 1 do begin
        // Handelt es sich bei der aktuellen Komponente um ein Label -> weitere Prfung
        if (obTempFrame.Components[iFrameIndex] is TLabel) then begin
          Result := PrvCheckLabel(obTempFrame.Components[iFrameIndex] as TLabel);
          if not Result then begin
            Exit;
          end; // if
        end; // if
      end; // for
    end; // else
  end; // for
end; // TCaoCheck.CheckData

//------------------------------------------------------------------------------


function TCaoCheck.CheckPassword(sPassword: string): Boolean;
(*
  berprft das eingegebene Passwort
*)
begin
  Result := sPassword = m_obRegistry.ReadPassword;
end; // TCaoCheck.CheckPassword

//------------------------------------------------------------------------------

function TCaoCheck.DeleteLettersInString(var sString: string; fDotAllowed: Boolean): Boolean;
(*
  berprft, ob alle Zeichen in einem String (sString) eine Zahl ist.
  Ist das nicht der Fall, so wird das falsche Zeichen gelscht und der
  Rckgabewert auf True gesetzt.
*)
var
  iIndex             : Integer;
  stTempAllowedChar  : Set of Char;
begin
  Result := False;
  stTempAllowedChar := c_stCharZahlen;
  if fDotAllowed then begin
  stTempAllowedChar := stTempAllowedChar + ['.'];
  end; // if
  if Length(sString) > 0 then begin
    for iIndex := 1 to Length(sString) do begin
      if not (sString[iIndex] in stTempAllowedChar) then begin
        Delete(sString, iIndex, 1);
        Result := True;
      end; // if
    end; // for
  end; // if
end; // TCaoCheck.DeleteLettersInString

//------------------------------------------------------------------------------

procedure TCaoCheck.EditWithNumbers(var obEdit: TEdit; fDotAllowed: Boolean);
(*
  Diese Prozedur lsst beim angegebenen Edit (obEdit) nur Zahlen als
  Eingaben zu. Andere Zeichen werden gelscht.
*)
var
  sTempString  : String;
  fDeleted     : Boolean;
  iTempSelStart: Integer;
begin
  // Der String muss zur berprfung in einem Temporren String
  // zwischengespeichert werden
  sTempString            := obEdit.Text;
  // Zwischenspeichern der Cursorposition
  iTempSelStart          := obEdit.SelStart;
  // String auf Zahlen berprfen
  fDeleted := DeleteLettersInString(sTempString, fDotAllowed);
  // Den Temporren String wieder zurckschreiben
  obEdit.Text       := sTempString;
  if fDeleted then begin
    // Zurcksetzen der Cursorposition
    obEdit.SelStart := iTempSelStart - 1;
  end; // if
end;

//------------------------------------------------------------------------------

procedure TCaoCheck.InitEditsInForms(var obForm: TForm);
(*
  Alle Editfelder eines Formulars leeren -> ''
*)
var
  iIndex,
  jIndex: Integer;
begin
  // Alle Komponenten des Formulars durchgehen
  for iIndex := 0 to obForm.ComponentCount - 1 do begin
    // Wird ein Edit gefunden, so wird der Text gelscht
    if (obForm.Components[iIndex] is TEdit) then begin
      (obForm.Components[iIndex] as TEdit).Clear;
    end; // if
    // Handelt es sich bei der gefundenen Komponente um ein Frame, so muss auch
    // dieses durchforstet werden
    if (obForm.Components[iIndex] is TFrame) then begin
      for jIndex := 0 to (obForm.Components[iIndex] as TFrame).ComponentCount - 1 do begin
        if ((obForm.Components[iIndex] as TFrame).Components[jIndex] is TEdit) then begin
          ((obForm.Components[iIndex] as TFrame).Components[jIndex] as TEdit).Clear;
        end; // if
      end; // for
    end; // if
  end; // for
end; // TCaoCheck.InitEditsInForms

//------------------------------------------------------------------------------

function TCaoCheck.ListViewItemIsSelected(obListView: TListView): Boolean;
(*
  berprft, ob in einer Listview ein Eintrag selektiert wurde
*)
begin
  Result := obListView.Selected <> nil;
  if not Result then begin
    MessageDlg(c_sMeldungKeineSelektion, mtError, [mbOk], 0);
  end; // if
end; // TCaoCheck.IsListViewItemSelected

//------------------------------------------------------------------------------

function TCaoCheck.PrvCheckLabel(obLabel: TLabel): Boolean;
(*
  berprft, ob es sich bei obLabel(Parameter) um ein Label handelt, dass zu
  einem Muss-Eingabe Editfeld gehrt. Ist das der Fall, so wird geprft, ob
  die Eingabe gemacht wurde. Das Ergebnis wird als Rckgabewert zurckgegeben.
  Wurde keine Eingabe gemacht, so wird eine Fehlermeldung ausgegeben.
*)
var
  sTempLabelCaption: string;
begin
  // Ist kein Label einem Edit-Feld zugeordnet wird True Zurckgegeben
  Result := True;
  // Prfen, ob es sich um ein Muss-Eingabefeld handelt
  if (obLabel.Caption[1] = '*') and (obLabel.FocusControl is TEdit) then begin
    Result := (obLabel.FocusControl as TEdit).Text <> '';

    // wurde keine Eingabe gemacht -> Fehlermeldung
    if not Result then begin
      sTempLabelCaption := obLabel.Caption;
      // & Aus dem Labeltext entfernen
      sTempLabelCaption := StringReplace(sTempLabelCaption, '&', '', []);
      // * Aus dem Labeltext entfernen
      sTempLabelCaption := StringReplace(sTempLabelCaption, '*', '', []);
      MessageDlg(Format(c_sMeldungEingabeErforderlich, [sTempLabelCaption]), mtError, [mbOk], 0);
      // Das Editfeld mit der fehlenden Eingabe soll den Fokus erhalten
      TEdit(obLabel.FocusControl).SetFocus;
    end; // if
  end; // if
end; // TCaoCheck.PrvCheckLabel

//------------------------------------------------------------------------------

constructor TCaoCheck.Create;
begin
  inherited Create;
  m_obRegistry := TCaoRegistry.Create;
end; // TCaoCheck.Create

//------------------------------------------------------------------------------

destructor TCaoCheck.Destroy;
begin
  m_obRegistry.Free;
  inherited Destroy;
end; // TCaoCheck.Destroy

//------------------------------------------------------------------------------

end. // CaoCheck
